import { EventEmitter } from "@dage/events";
export class EventBus {
    constructor() {
        this.emitter = new EventEmitter();
        this.bufferMap = new Map();
    }
    getBufferValue(event) {
        return this.bufferMap.get(event);
    }
    setBufferValue(event, val) {
        this.bufferMap.set(event, val);
    }
    /**
     * 监听事件
     */
    on(event, callback) {
        const bufferValue = this.getBufferValue(event);
        if (bufferValue) {
            callback(bufferValue);
        }
        this.emitter.on(event, callback);
        return () => {
            this.emitter.off(event, callback);
        };
    }
    /**
     * 监听一次事件
     */
    once(event, callback) {
        const bufferValue = this.getBufferValue(event);
        if (bufferValue) {
            callback(bufferValue);
            return () => { };
        }
        this.emitter.once(event, callback);
        return () => {
            this.emitter.off(event, callback);
        };
    }
    /**
     * 触发事件
     * @param event
     * @param arg
     * @param replay - 是否重放该值
     */
    emit(event, arg, replay = false) {
        if (replay) {
            this.setBufferValue(event, arg);
        }
        this.emitter.emit(event, arg);
    }
    /**
     * 需要最新值的时候 可以调用此方法重置
     */
    resetReplayValue(event) {
        this.bufferMap.delete(event);
    }
    /**
     * 监听新的订阅者
     */
    onSubscriber(event, callback) {
        const listener = (name, subscriber) => {
            if (name === event) {
                callback(subscriber);
            }
        };
        this.emitter.on("newListener", listener);
        return () => {
            this.emitter.off("newListener", listener);
        };
    }
}
