"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatDate = exports.parseDate = exports.DAY = exports.HOUR = exports.MINUTE = exports.SECOND = exports.DATE_TIME_FORMAT = exports.DATE_FORMAT = exports.dayjs = void 0;
const tslib_1 = require("tslib");
/**
 * 日期和时间相关
 *
 * 更复杂的日期操作建议直接使用 dayjs https://dayjs.gitee.io/docs/zh-CN/installation/installation
 */
const dayjs_1 = tslib_1.__importDefault(require("dayjs"));
exports.dayjs = dayjs_1.default;
const customParseFormat_1 = tslib_1.__importDefault(require("dayjs/plugin/customParseFormat"));
dayjs_1.default.extend(customParseFormat_1.default);
/**
 * @group 日期格式
 * @groupOrder 4
 */
exports.DATE_FORMAT = 'YYYY-MM-DD';
/**
 * @group 日期格式
 * @groupOrder 4
 */
exports.DATE_TIME_FORMAT = 'YYYY-MM-DD HH:mm:ss';
exports.SECOND = 1000; // 毫秒
exports.MINUTE = 60 * exports.SECOND;
exports.HOUR = 60 * exports.MINUTE;
exports.DAY = 24 * exports.HOUR;
/**
 * 日期解析
 *
 * @group 日期
 * @groupOrder 5
 *
 * @remarks
 * 如果传入数字，我们假定为`毫秒` <br/>
 *` parseDate`会先尝试常见的日期格式（例如 YYYY/MM/DD HH:mm:ss、YYYY-MM-DD HH:mm:ss、YYYY-MM-DD），如果`parse`失败就会使用`format`
 *
 * @param dateString - 要转换成日期的数字和字符串.
 * @param format - 用于指定日期的格式。如果你的日期格式不是常见的格式，可以显式指定
 * @link https://dayjs.gitee.io/docs/zh-CN/parse/string-format
 *
 * @example
 * 例子:
 * ```typescript
 * parseDate(3600)
 * // => 1970-01-01T00:00:03.600Z
 *
 * parseDate('2012/12/10 12:12:12')
 * // => 2012-02-10T04:12:12.000Z
 * ```
 * @public
 */
function parseDate(dateString, format) {
    if (typeof dateString === 'number') {
        return new Date(dateString);
    }
    else if (dateString instanceof Date) {
        return dateString;
    }
    const day = (0, dayjs_1.default)(dateString, format);
    if (day.isValid()) {
        return day.toDate();
    }
    else {
        if (!format) {
            // 兼容 ios
            dateString = dateString.replace(/-/g, '/');
            dateString = dateString.replace(/T/g, ' ');
        }
        const retry = (0, dayjs_1.default)(dateString);
        if (retry.isValid()) {
            return retry.toDate();
        }
        return null;
    }
}
exports.parseDate = parseDate;
/**
 * 格式化日期
 *
 * @group 日期
 * @groupOrder 5
 *
 * @remarks
 * 如果传入数字，我们假定为`毫秒`
 *
 * @param date - 日期.
 * @param format - 详见 {@link https://dayjs.gitee.io/docs/zh-CN/parse/string-format}.
 *
 * @example
 * 例子:
 * ```
 * const day = new Date('2012/12/12 12:12:12');
 *
 * const target = formatDate(day);
 * // => target = "2012-02-10"
 *
 * const target2 = formatDate(day, 'YYYY-MM-DD HH:mm:ss');
 * // => target2 = "2012-02-10 12:12:12"
 * ```
 * @public
 */
function formatDate(date, format = exports.DATE_FORMAT, parseFormat) {
    const parsed = parseDate(date, parseFormat);
    return (0, dayjs_1.default)(parsed).format(format);
}
exports.formatDate = formatDate;
